/**
  ******************************************************************************
  * @file    py32t020_ll_rcc.h
  * @author  MCU Application Team
  * @brief   Header file of RCC LL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32T020_LL_RCC_H
#define __PY32T020_LL_RCC_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32t0xx.h"

/** @addtogroup PY32T020_LL_Driver
  * @{
  */

#if defined(RCC)

/** @defgroup RCC_LL RCC
  * @{
  */

/* Private types -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/** @defgroup RCC_LL_Private_Variables RCC Private Variables
  * @{
  */
/**
  * @}
  */

/* Private constants ---------------------------------------------------------*/
/* Private macros ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_Private_Macros RCC Private Macros
  * @{
  */
/**
  * @}
  */
#endif /*USE_FULL_LL_DRIVER*/

/* Exported types ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_Exported_Types RCC Exported Types
  * @{
  */

/** @defgroup LL_ES_CLOCK_FREQ Clocks Frequency Structure
  * @{
  */

/**
  * @brief  RCC Clocks Frequency Structure
  */
typedef struct
{
  uint32_t SYSCLK_Frequency;        /*!< SYSCLK clock frequency */
  uint32_t HCLK_Frequency;          /*!< HCLK clock frequency */
  uint32_t PCLK1_Frequency;         /*!< PCLK1 clock frequency */
} LL_RCC_ClocksTypeDef;

/**
  * @}
  */

/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/* Exported constants --------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Constants RCC Exported Constants
  * @{
  */

/** @defgroup RCC_LL_EC_OSC_VALUES Oscillator Values adaptation
  * @brief    Defines used to adapt values of different oscillators
  * @note     These values could be modified in the user environment according to
  *           HW set-up.
  * @{
  */
#if !defined  (HSE_VALUE)// todo
#define HSE_VALUE    8000000U   /*!< Value of the HSE oscillator in Hz */
#endif /* HSE_VALUE */

#if !defined  (HSI_VALUE)// todo
#define HSI_VALUE    24000000U  /*!< Value of the HSI oscillator in Hz */
#endif /* HSI_VALUE */

#if defined(RCC_LSE_SUPPORT)
#if !defined  (LSE_VALUE)
#define LSE_VALUE    32768U     /*!< Value of the LSE oscillator in Hz */
#endif /* LSE_VALUE */
#endif

#if !defined  (LSI_VALUE)
#define LSI_VALUE    32768U     /*!< Value of the LSI oscillator in Hz */
#endif /* LSI_VALUE */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_CLEAR_FLAG Clear Flags Defines
  * @brief    Flags defines which can be used with LL_RCC_WriteReg function
  * @{
  */
#define LL_RCC_CICR_LSIRDYC                RCC_CICR_LSIRDYC     /*!< LSI Ready Interrupt Clear */
#define LL_RCC_CICR_HSIRDYC                RCC_CICR_HSIRDYC     /*!< HSI Ready Interrupt Clear */
#define LL_RCC_CICR_HSERDYC                RCC_CICR_HSERDYC     /*!< HSE Ready Interrupt Clear */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CICR_LSERDYC                RCC_CICR_LSERDYC     /*!< LSE Ready Interrupt Clear */
#define LL_RCC_CICR_LSECSSC                RCC_CICR_LSECSSC     /*!< LSE Clock Security System Interrupt Clear */
#endif
#define LL_RCC_CICR_CSSC                   RCC_CICR_CSSC        /*!< Clock Security System Interrupt Clear */

/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_GET_FLAG Get Flags Defines
  * @brief    Flags defines which can be used with LL_RCC_ReadReg function
  * @{
  */
#define LL_RCC_CIFR_LSIRDYF                RCC_CIFR_LSIRDYF     /*!< LSI Ready Interrupt flag */
#define LL_RCC_CIFR_HSIRDYF                RCC_CIFR_HSIRDYF     /*!< HSI Ready Interrupt flag */
#define LL_RCC_CIFR_HSERDYF                RCC_CIFR_HSERDYF     /*!< HSE Ready Interrupt flag */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CIFR_LSERDYF                RCC_CIFR_LSERDYF     /*!< LSE Ready Interrupt flag */
#define LL_RCC_CIFR_LSECSSF                RCC_CIFR_LSECSSF     /*!< LSE Clock Security System Interrupt flag */
#endif
#define LL_RCC_CIFR_CSSF                   RCC_CIFR_CSSF        /*!< Clock Security System Interrupt flag */
// todo
#define LL_RCC_CSR_BORRSTF                 RCC_CSR_BORRSTF    /*!< BOR reset flag */
#define LL_RCC_CSR_OBLRSTF                 RCC_CSR_OBLRSTF    /*!< OBL reset flag */
#define LL_RCC_CSR_PINRSTF                 RCC_CSR_PINRSTF    /*!< PIN reset flag */
#define LL_RCC_CSR_SFTRSTF                 RCC_CSR_SFTRSTF    /*!< Software Reset flag */
#define LL_RCC_CSR_IWDGRSTF                RCC_CSR_IWDGRSTF   /*!< Independent Watchdog reset flag */
#define LL_RCC_CSR_PORRSTF                 RCC_CSR_PORRSTF    /*!< BOR or POR/PDR reset flag */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_IT IT Defines
  * @brief    IT defines which can be used with LL_RCC_ReadReg and  LL_RCC_WriteReg functions
  * @{
  */
#define LL_RCC_CIER_LSIRDYIE               RCC_CIER_LSIRDYIE      /*!< LSI Ready Interrupt Enable */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CIER_LSERDYIE               RCC_CIER_LSERDYIE      /*!< LSE Ready Interrupt Enable */
#endif
#define LL_RCC_CIER_HSIRDYIE               RCC_CIER_HSIRDYIE      /*!< HSI Ready Interrupt Enable */
#define LL_RCC_CIER_HSERDYIE               RCC_CIER_HSERDYIE      /*!< HSE Ready Interrupt Enable */
/**
  * @}
  */
  
#if defined(RCC_LSE_SUPPORT)
/** @defgroup RCC_LL_EC_LSEDRIVE  LSE oscillator drive capability
  * @{
  */
#define LL_RCC_LSEDRIVE_CLOSE              0x00000000                         /*!< LSE driving capability closed */
#define LL_RCC_LSEDRIVE_LOW                RCC_ECSCR_LSE_DRIVER_0             /*!< LSE lower driving capability */
#define LL_RCC_LSEDRIVE_MEDIUM             RCC_ECSCR_LSE_DRIVER_1             /*!< LSE medium driving capability */
#define LL_RCC_LSEDRIVE_HIGH               RCC_ECSCR_LSE_DRIVER               /*!< LSE higher driving capability */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_LSESTARTUP  LSE oscillator startup time
  * @{
  */
#define LL_RCC_LSESTARTUP_DELAY_LOW              RCC_ECSCR_LSE_STARTUP         /*!< LSE Startup none delay */
#define LL_RCC_LSESTARTUP_DELAY_MEDIUM           RCC_ECSCR_LSE_STARTUP_0       /*!< LSE Startup short delay */
#define LL_RCC_LSESTARTUP_DELAY_HIGH             0x00000000                    /*!< LSE Startup long delay */
#define LL_RCC_LSESTARTUP_DELAY_VERY_HIGH        RCC_ECSCR_LSE_STARTUP_1       /*!< LSE Startup very long delay */
/**
  * @}
  */
#endif
// todo
#if defined(RCC_BDCR_LSCSEL)
/** @defgroup RCC_LL_EC_LSC_CLKSOURCE  LSC Selection
  * @{
  */
#define LL_RCC_LSC_CLKSOURCE_LSI          0x00000000U                 /*!< LSI selection for low speed clock  */
#define LL_RCC_LSC_CLKSOURCE_LSE          RCC_BDCR_LSCSEL            /*!< LSE selection for low speed clock  */
/**
  * @}
  */
#endif
// todo
/** @defgroup RCC_LL_EC_SYS_CLKSOURCE  System clock switch
  * @{
  */
#define LL_RCC_SYS_CLKSOURCE_HSISYS        0x00000000U                        /*!< HSISYS selection as system clock */
#define LL_RCC_SYS_CLKSOURCE_HSE           RCC_CFGR_SW_0                      /*!< HSE selection as system clock */
#define LL_RCC_SYS_CLKSOURCE_LSI           (RCC_CFGR_SW_1 | RCC_CFGR_SW_0)    /*!< LSI selection used as system clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_LSE           RCC_CFGR_SW_2                      /*!< LSE selection used as system clock */
#endif
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_SYS_CLKSOURCE_STATUS  System clock switch status
  * @{
  */
#define LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS 0x00000000U                         /*!< HSISYS used as system clock */
#define LL_RCC_SYS_CLKSOURCE_STATUS_HSE    RCC_CFGR_SWS_0                      /*!< HSE used as system clock */
#define LL_RCC_SYS_CLKSOURCE_STATUS_LSI    (RCC_CFGR_SWS_1 | RCC_CFGR_SWS_0)   /*!< LSI used as system clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_STATUS_LSE    RCC_CFGR_SWS_2                      /*!< LSE used as system clock */
#endif
/**
  * @}
  */

/** @defgroup RCC_LL_EC_SYSCLK_DIV  AHB prescaler
  * @{
  */
#define LL_RCC_SYSCLK_DIV_1                0x00000000U                                                             /*!< SYSCLK not divided */
#define LL_RCC_SYSCLK_DIV_2                RCC_CFGR_HPRE_3                                                         /*!< SYSCLK divided by 2 */
#define LL_RCC_SYSCLK_DIV_4                (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_0)                                     /*!< SYSCLK divided by 4 */
#define LL_RCC_SYSCLK_DIV_8                (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_1)                                     /*!< SYSCLK divided by 8 */
#define LL_RCC_SYSCLK_DIV_16               (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_1 | RCC_CFGR_HPRE_0)                   /*!< SYSCLK divided by 16 */
#define LL_RCC_SYSCLK_DIV_64               (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2)                                     /*!< SYSCLK divided by 64 */
#define LL_RCC_SYSCLK_DIV_128              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_0)                   /*!< SYSCLK divided by 128 */
#define LL_RCC_SYSCLK_DIV_256              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_1)                   /*!< SYSCLK divided by 256 */
#define LL_RCC_SYSCLK_DIV_512              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_1 | RCC_CFGR_HPRE_0) /*!< SYSCLK divided by 512 */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_APB1_DIV  APB1 low-speed prescaler (APB1)
  * @{
  */
#define LL_RCC_APB1_DIV_1                  0x00000000U                                           /*!< HCLK not divided */
#define LL_RCC_APB1_DIV_2                  RCC_CFGR_PPRE_2                                       /*!< HCLK divided by 2 */
#define LL_RCC_APB1_DIV_4                  (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_0)                   /*!< HCLK divided by 4 */
#define LL_RCC_APB1_DIV_8                  (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_1)                   /*!< HCLK divided by 8 */
#define LL_RCC_APB1_DIV_16                 (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_1 | RCC_CFGR_PPRE_0) /*!< HCLK divided by 16 */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_HSI_DIV  HSI division factor
  * @{
  */
#define LL_RCC_HSI_DIV_1                  0x00000000U                                /*!< HSI not divided */
#define LL_RCC_HSI_DIV_2                  RCC_CR_HSIDIV_0                            /*!< HSI divided by 2 */
#define LL_RCC_HSI_DIV_4                  RCC_CR_HSIDIV_1                            /*!< HSI divided by 4 */
#define LL_RCC_HSI_DIV_8                  (RCC_CR_HSIDIV_1 | RCC_CR_HSIDIV_0)        /*!< HSI divided by 8 */
#define LL_RCC_HSI_DIV_16                 RCC_CR_HSIDIV_2                            /*!< HSI divided by 16 */
#define LL_RCC_HSI_DIV_32                 (RCC_CR_HSIDIV_2 | RCC_CR_HSIDIV_0)        /*!< HSI divided by 32 */
#define LL_RCC_HSI_DIV_64                 (RCC_CR_HSIDIV_2 | RCC_CR_HSIDIV_1)        /*!< HSI divided by 64 */
#define LL_RCC_HSI_DIV_128                RCC_CR_HSIDIV                              /*!< HSI divided by 128 */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_MCOSOURCE  MCO SOURCE selection
  * @{
  */
#define LL_RCC_MCOSOURCE_NOCLOCK         0x00000000U                                                 /*!< MCO output disabled, no clock on MCO */
#define LL_RCC_MCOSOURCE_SYSCLK          RCC_CFGR_MCOSEL_0                                           /*!< SYSCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_HSI10M          RCC_CFGR_MCOSEL_1                                           /*!< HSI10M selection as MCO source */
#define LL_RCC_MCOSOURCE_HSI             (RCC_CFGR_MCOSEL_1 | RCC_CFGR_MCOSEL_0)                     /*!< HSI selection as MCO source */
#define LL_RCC_MCOSOURCE_HSE             RCC_CFGR_MCOSEL_2                                           /*!< HSE selection as MCO source */
#define LL_RCC_MCOSOURCE_LSI             (RCC_CFGR_MCOSEL_2 | RCC_CFGR_MCOSEL_1)                     /*!< LSI selection as MCO source */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_MCOSOURCE_LSE             (RCC_CFGR_MCOSEL_2 | RCC_CFGR_MCOSEL_1 | RCC_CFGR_MCOSEL_0) /*!< LSE selection as MCO source */
#endif
#define LL_RCC_MCOSOURCE_HCLK            RCC_CFGR_MCOSEL_3                                           /*!< HCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_PCLK            (RCC_CFGR_MCOSEL_3 | RCC_CFGR_MCOSEL_0)                     /*!< PCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_HCLK_SRC        (RCC_CFGR_MCOSEL_3 | RCC_CFGR_MCOSEL_1)                     /*!< HCLK without gate selected as MCO source */
#define LL_RCC_MCOSOURCE_PCLK_UNGATE     (RCC_CFGR_MCOSEL_3 | RCC_CFGR_MCOSEL_1 | RCC_CFGR_MCOSEL_0) /*!< PCLK without gate selected as MCO source */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_MCO Peripheral MCO get clock source
  * @{
  */
#define LL_RCC_MCO_CLKSOURCE             RCC_CFGR_MCOSEL /*!< MCO Clock source selection */
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EC_MCO_DIV  MCO prescaler
  * @{
  */
#define LL_RCC_MCO_DIV_1                  0x00000000U                                                 /*!< MCO not divided */
#define LL_RCC_MCO_DIV_2                  RCC_CFGR_MCOPRE_0                                           /*!< MCO divided by 2 */
#define LL_RCC_MCO_DIV_4                  RCC_CFGR_MCOPRE_1                                           /*!< MCO divided by 4 */
#define LL_RCC_MCO_DIV_8                  (RCC_CFGR_MCOPRE_1 | RCC_CFGR_MCOPRE_0)                     /*!< MCO divided by 8 */
#define LL_RCC_MCO_DIV_16                 RCC_CFGR_MCOPRE_2                                           /*!< MCO divided by 16 */
#define LL_RCC_MCO_DIV_32                 (RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_0)                     /*!< MCO divided by 32 */
#define LL_RCC_MCO_DIV_64                 (RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_1)                     /*!< MCO divided by 64 */
#define LL_RCC_MCO_DIV_128                (RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_1 | RCC_CFGR_MCOPRE_0) /*!< MCO divided by 128 */
/**
  * @}
  */

#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_EC_PERIPH_FREQUENCY Peripheral clock frequency
  * @{
  */
#define LL_RCC_PERIPH_FREQUENCY_NO        0x00000000U                 /*!< No clock enabled for the peripheral            */
#define LL_RCC_PERIPH_FREQUENCY_NA        0xFFFFFFFFU                 /*!< Frequency cannot be provided as external clock */
/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */
// todo
#if defined(RCC_CCIPR_TIMCLK_CTRL)
/** @defgroup RCC_LL_EC_TIM_PCLK_Frequency_Control TIM PCLK Frequency Control
  * @{
  */
#define LL_RCC_TIMPCLK_MUL2               0x00000000U            /*!< TIMER PCLK is twice the system PCLK, but the frequency will not exceed HCLK */
#define LL_RCC_TIMPCLK_MUL1               RCC_CCIPR_TIMCLK_CTRL  /*!< TIMER PCLK is the system PCLK */
/**
  * @}
  */
#endif /* RCC_CCIPR_TIM1SEL */

#if defined(COMP1)
/** @defgroup RCC_LL_EC_COMPx_CLKSOURCE Peripheral COMP clock source selection
  * @{
  */
#define LL_RCC_COMP1_CLKSOURCE_PCLK1      (RCC_CCIPR_COMP1SEL | (0x00000000U >> 8U))                    /*!< PCLK1 selected as COMP1 clock */
#define LL_RCC_COMP1_CLKSOURCE_LSC        (RCC_CCIPR_COMP1SEL | (RCC_CCIPR_COMP1SEL >> 8U))             /*!< LSC selected as COMP1 clock */
#if defined(COMP2)
#define LL_RCC_COMP2_CLKSOURCE_PCLK1      (RCC_CCIPR_COMP2SEL | (0x00000000U >> 8U))                    /*!< PCLK1 selected as COMP2 clock */
#define LL_RCC_COMP2_CLKSOURCE_LSC        (RCC_CCIPR_COMP2SEL | (RCC_CCIPR_COMP2SEL >> 8U))             /*!< LSC selected as COMP2 clock */
#endif
/**
  * @}
  */
#endif /* COMP1 && COMP2 */
// todo
#if defined(RCC_CCIPR_IWDGSEL)
/** @defgroup RCC_LL_EC_IWDG_CLKSOURCE Peripheral IWDG clock source selection
  * @{
  */
#define LL_RCC_IWDG_CLKSOURCE_LSI        0x00000000U        /*!< LSI selected as IWDG clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_IWDG_CLKSOURCE_LSE        RCC_CCIPR_IWDGSEL  /*!< LSE selected as IWDG clock */
#endif
/**
  * @}
  */
#endif /* RCC_CCIPR_IWDGSEL */
// todo
#if defined(RCC_BDCR_RTCSEL)
/** @defgroup RCC_LL_EC_RTC_CLKSOURCE  RTC clock source selection
  * @{
  */
#define LL_RCC_RTC_CLKSOURCE_NONE          0x00000000U                                                        /*!< No clock used as RTC clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_RTC_CLKSOURCE_LSE         RCC_BDCR_RTCSEL_0                                                    /*!< LSE oscillator clock used as RTC clock */
#endif
#define LL_RCC_RTC_CLKSOURCE_LSI         RCC_BDCR_RTCSEL_1                                                    /*!< LSI oscillator clock used as RTC clock */
#define LL_RCC_RTC_CLKSOURCE_HSE_DIV32   (                            RCC_BDCR_RTCSEL_1 | RCC_BDCR_RTCSEL_0)  /*!< HSE oscillator clock divided by 32 used as RTC clock */
#define LL_RCC_RTC_CLKSOURCE_HSE_DIV128  (RCC_BDCR_RTC_HSEDIV_SEL_0 | RCC_BDCR_RTCSEL_1 | RCC_BDCR_RTCSEL_0)  /*!< HSE oscillator clock divided by 128 used as RTC clock */
#define LL_RCC_RTC_CLKSOURCE_HSE_DIV8    (RCC_BDCR_RTC_HSEDIV_SEL_1 | RCC_BDCR_RTCSEL_1 | RCC_BDCR_RTCSEL_0)  /*!< HSE oscillator clock divided by 8 used as RTC clock */
/**
  * @}
  */
#endif
// todo
#if defined(RCC_CCIPR_TIMCLK_CTRL)
/** @defgroup RCC_LL_EC_TIMCLK_Control Peripheral TIM PCLK Frequency Control
  * @{
  */
#define LL_RCC_TIMPCLK_CTRL                RCC_CCIPR_TIMCLK_CTRL /*!< TIM PCLK Frequency Control */
/**
  * @}
  */
#endif

#if defined(COMP1)
/** @defgroup RCC_LL_EC_COMP Peripheral COMP get clock source
  * @{
  */
#define LL_RCC_COMP1_CLKSOURCE            RCC_CCIPR_COMP1SEL /*!< COMP1 Clock source selection */
#if defined(COMP2)
#define LL_RCC_COMP2_CLKSOURCE            RCC_CCIPR_COMP2SEL /*!< COMP2 Clock source selection */
#endif /* COMP2 */
/**
  * @}
  */
#endif /* COMP1 */
// todo
#if defined(RCC_CCIPR_IWDGSEL)
/** @defgroup RCC_LL_EC_IWDG Peripheral IWDG get clock source
  * @{
  */
#define LL_RCC_IWDG_CLKSOURCE            RCC_CCIPR_IWDGSEL /*!< IWDG Clock source selection */
/**
  * @}
  */
#endif /* RCC_CCIPR_IWDGSEL */
// todo
/** @defgroup RCC_HSE_EC_Freq HSE Config
  * @{
  */
#define LL_RCC_HSE_4_6MHz                 0x00000000U
#define LL_RCC_HSE_6_8MHz                 RCC_ECSCR_HSE_DRV
/**
  * @}
  */
// todo
/** @defgroup RCC_HSE_EC_STARTUP HSE settling time Config
  * @{
  */
#define LL_RCC_HSE_STARTUP_NONE         (RCC_ECSCR_HSE_STARTUP_1 | RCC_ECSCR_HSE_STARTUP_0) /* Direct output regardless of stabilization time. */
#define LL_RCC_HSE_STARTUP_LOW          RCC_ECSCR_HSE_STARTUP_0                             /* It is output after 2048 HSE clock cycles. 
                                                                                               If HSEBYP is set, it is output after 1024 clock cycles. */
#define LL_RCC_HSE_STARTUP_MEDIUM       0x00000000U                                         /* It is output after 4096 HSE clock cycles.
                                                                                               If HSEBYP is set, it is output after 2048 clock cycles. */
#define LL_RCC_HSE_STARTUP_HIGH         RCC_ECSCR_HSE_STARTUP_1                             /* It is output after 8192 HSE clock cycles.
                                                                                               If HSEBYP is set, it is output after 4096 clock cycles. */
/**
  * @}
  */
// todo
/** @defgroup RCC_HSI_EC_Calibration HSI Calibration
  * @{
  */
#define LL_RCC_HSICALIBRATION_24MHz       ((0x4<<13) | ((*(uint32_t *)(0x1FFF0100)) & 0x1FFF))  /*!< 24MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_48MHz       ((0x5<<13) | ((*(uint32_t *)(0x1FFF0104)) & 0x1FFF))  /*!< 48MHz HSI calibration trimming value */
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Macros RCC Exported Macros
  * @{
  */

/** @defgroup RCC_LL_EM_WRITE_READ Common Write and read registers Macros
  * @{
  */

/**
  * @brief  Write a value in RCC register
  * @param  __REG__ Register to be written
  * @param  __VALUE__ Value to be written in the register
  * @retval None
  */
#define LL_RCC_WriteReg(__REG__, __VALUE__) WRITE_REG((RCC->__REG__), (__VALUE__))

/**
  * @brief  Read a value in RCC register
  * @param  __REG__ Register to be read
  * @retval Register value
  */
#define LL_RCC_ReadReg(__REG__) READ_REG(RCC->__REG__)
/**
  * @}
  */

/** @defgroup RCC_LL_EM_CALC_FREQ Calculate frequencies
  * @{
  */
// todo
/**
  * @brief  Helper macro to calculate the HCLK frequency
  * @param  __SYSCLKFREQ__ SYSCLK frequency
  * @param  __AHBPRESCALER__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  * @retval HCLK clock frequency (in Hz)
  */
#define __LL_RCC_CALC_HCLK_FREQ(__SYSCLKFREQ__,__AHBPRESCALER__)  \
  ((__SYSCLKFREQ__) >> (AHBPrescTable[((__AHBPRESCALER__) & RCC_CFGR_HPRE) >>  RCC_CFGR_HPRE_Pos] & 0x1FU))

/**
  * @brief  Helper macro to calculate the PCLK1 frequency (APB1)
  * @param  __HCLKFREQ__ HCLK frequency
  * @param  __APB1PRESCALER__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  * @retval PCLK1 clock frequency (in Hz)
  */
#define __LL_RCC_CALC_PCLK1_FREQ(__HCLKFREQ__, __APB1PRESCALER__)  \
  ((__HCLKFREQ__) >> (APBPrescTable[(__APB1PRESCALER__) >>  RCC_CFGR_PPRE_Pos] & 0x1FU))

/**
  * @brief  Helper macro to calculate the HSISYS frequency
  * @param  __HSIDIV__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  * @retval HSISYS clock frequency (in Hz)
  */
#define __LL_RCC_CALC_HSI_FREQ(__HSIDIV__)  \
  (HSIFreqTable[(RCC->ICSCR & RCC_ICSCR_HSI_FS) >> RCC_ICSCR_HSI_FS_Pos] / (1U << ((__HSIDIV__)>> RCC_CR_HSIDIV_Pos)))

/**
  * @}
  */

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Functions RCC Exported Functions
  * @{
  */

/** @defgroup RCC_LL_EF_HSE HSE
  * @{
  */
// todo
/**
  * @brief  Enable the Clock Security System.
  * @rmtoll CR           HSE_CSSON         LL_RCC_HSE_EnableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_EnableCSS(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSE_CSSON);
}

/**
  * @brief  Enable HSE external oscillator (HSE Bypass)
  * @rmtoll CR           HSEBYP        LL_RCC_HSE_EnableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_EnableBypass(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSEBYP);
}

/**
  * @brief  Disable HSE external oscillator (HSE Bypass)
  * @rmtoll CR           HSEBYP        LL_RCC_HSE_DisableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_DisableBypass(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSEBYP);
}

/**
  * @brief  Enable HSE crystal oscillator (HSE ON)
  * @rmtoll CR           HSEON         LL_RCC_HSE_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSEON);
}

/**
  * @brief  Disable HSE crystal oscillator (HSE ON)
  * @rmtoll CR           HSEON         LL_RCC_HSE_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSEON);
}

/**
  * @brief  Check if HSE oscillator Ready
  * @rmtoll CR           HSERDY        LL_RCC_HSE_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_HSE_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_HSERDY) == (RCC_CR_HSERDY)) ? 1UL : 0UL);
}
// todo
/**
  * @brief  Set HSE Crystal working frequency
  * @param  HSEFreq This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSE_4_6MHz
  *         @arg @ref LL_RCC_HSE_6_8MHz
  * @note   The maximum working frequency varies depending on the devices.
  *         Refer to the devices data sheet for the maximum working frequency.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_SetFreqRegion(uint32_t HSEFreq)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_HSE_DRV, HSEFreq);
}
// todo
/**
  * @brief  Set HSE settling time.
  * @param  Time This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSE_STARTUP_NONE
  *         @arg @ref LL_RCC_HSE_STARTUP_LOW
  *         @arg @ref LL_RCC_HSE_STARTUP_MEDIUM
  *         @arg @ref LL_RCC_HSE_STARTUP_HIGH
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_SetStartupDelay(uint32_t Time)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_HSE_STARTUP, Time);
}
// todo
/**
  * @brief  Get HSE settling time.
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_HSE_STARTUP_NONE
  *         @arg @ref LL_RCC_HSE_STARTUP_LOW
  *         @arg @ref LL_RCC_HSE_STARTUP_MEDIUM
  *         @arg @ref LL_RCC_HSE_STARTUP_HIGH
  */
__STATIC_INLINE uint32_t LL_RCC_HSE_GetStartupDelay(void)
{
  return (uint32_t)(READ_BIT(RCC->ECSCR, RCC_ECSCR_HSE_STARTUP));
}

/**
  * @}
  */

/** @defgroup RCC_LL_EF_HSI HSI
  * @{
  */

/**
  * @brief  Enable HSI oscillator
  * @rmtoll CR           HSION         LL_RCC_HSI_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSION);
}

/**
  * @brief  Disable HSI oscillator
  * @rmtoll CR           HSION         LL_RCC_HSI_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSION);
}

/**
  * @brief  Check if HSI clock is ready
  * @rmtoll CR           HSIRDY        LL_RCC_HSI_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_HSIRDY) == (RCC_CR_HSIRDY)) ? 1UL : 0UL);
}

/**
  * @brief  Set HSI Calibration trimming
  * @param  Value Between Min_Data = 0 and Max_Data = 0x1FFF
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_SetCalibTrimming(uint32_t Value)
{
  MODIFY_REG(RCC->ICSCR, RCC_ICSCR_HSI_TRIM, Value << RCC_ICSCR_HSI_TRIM_Pos);
}

/**
  * @brief  Get HSI Calibration trimming
  * @rmtoll ICSCR        HSITRIM       LL_RCC_HSI_GetCalibTrimming
  * @retval Between Min_Data = 0 and Max_Data = 0x1FFF
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_GetCalibTrimming(void)
{
  return (uint32_t)(READ_BIT(RCC->ICSCR, RCC_ICSCR_HSI_TRIM) >> RCC_ICSCR_HSI_TRIM_Pos);
}

/**
  * @brief  Set HSI Calibration Frequency
  * @param  Value This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSICALIBRATION_24MHz
  *         @arg @ref LL_RCC_HSICALIBRATION_48MHz
  * @note   Depending on devices and packages, some calibration values may not be available.
  *         Refer to device datasheet for calibration values availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_SetCalibFreq(uint32_t Value)
{
  MODIFY_REG(RCC->ICSCR, (RCC_ICSCR_HSI_FS | RCC_ICSCR_HSI_TRIM), Value);
}

/**
  * @brief  Get HSI Frequency
  * @retval HSI clock frequency (in Hz)
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_GetFreq(void)
{
  return (uint32_t)HSIFreqTable[(RCC->ICSCR & RCC_ICSCR_HSI_FS) >> RCC_ICSCR_HSI_FS_Pos];
}

/**
  * @}
  */

#if defined(RCC_LSE_SUPPORT)
/** @defgroup RCC_LL_EF_LSE LSE
  * @{
  */

/**
  * @brief  Enable  Low Speed External (LSE) crystal.
  * @rmtoll BDCR         LSEON         LL_RCC_LSE_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_Enable(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSEON);
}

/**
  * @brief  Disable  Low Speed External (LSE) crystal.
  * @rmtoll BDCR         LSEON         LL_RCC_LSE_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_Disable(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSEON);
}

/**
  * @brief  Enable external clock source (LSE bypass).
  * @rmtoll BDCR         LSEBYP        LL_RCC_LSE_EnableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_EnableBypass(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSEBYP);
}

/**
  * @brief  Disable external clock source (LSE bypass).
  * @rmtoll BDCR         LSEBYP        LL_RCC_LSE_DisableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_DisableBypass(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSEBYP);
}

/**
  * @brief  Set LSE oscillator drive capability
  * @note The oscillator is in Xtal mode when it is not in bypass mode.
  * @param  LSEDrive This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSEDRIVE_CLOSE
  *         @arg @ref LL_RCC_LSEDRIVE_LOW
  *         @arg @ref LL_RCC_LSEDRIVE_MEDIUM
  *         @arg @ref LL_RCC_LSEDRIVE_HIGH
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_SetDriveCapability(uint32_t LSEDrive)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_LSE_DRIVER, LSEDrive);
}

/**
  * @brief  Get LSE oscillator drive capability
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSEDRIVE_CLOSE
  *         @arg @ref LL_RCC_LSEDRIVE_LOW
  *         @arg @ref LL_RCC_LSEDRIVE_MEDIUM
  *         @arg @ref LL_RCC_LSEDRIVE_HIGH
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_GetDriveCapability(void)
{
  return (uint32_t)(READ_BIT(RCC->ECSCR, RCC_ECSCR_LSE_DRIVER));
}
// todo
/**
  * @brief  Set LSE startup time
  * @param  LSEStartup This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_LOW      
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_MEDIUM   
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_HIGH     
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_VERY_HIGH
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_SetStartupTime(uint32_t LSEStartup)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_LSE_STARTUP, LSEStartup);
}
// todo
/**
  * @brief  Get LSE startup time
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_LOW      
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_MEDIUM   
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_HIGH     
  *         @arg @ref LL_RCC_LSESTARTUP_DELAY_VERY_HIGH
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_GetStartupTime(void)
{
  return (uint32_t)(READ_BIT(RCC->ECSCR, RCC_ECSCR_LSE_STARTUP));
}
/**
  * @brief  Enable Clock security system on LSE.
  * @rmtoll BDCR         LSECSSON      LL_RCC_LSE_EnableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_EnableCSS(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSECSSON);
}

/**
  * @brief  Disable Clock security system on LSE.
  * @note Clock security system can be disabled only after a LSE
  *       failure detection. In that case it MUST be disabled by software.
  * @rmtoll BDCR         LSECSSON      LL_RCC_LSE_DisableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_DisableCSS(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSECSSON);
}


/**
  * @brief  Check if LSE oscillator Ready
  * @rmtoll BDCR         LSERDY        LL_RCC_LSE_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_IsReady(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_LSERDY) == (RCC_BDCR_LSERDY)) ? 1UL : 0UL);
}

/**
  * @brief  Check if CSS on LSE failure Detection
  * @rmtoll BDCR         LSECSSD       LL_RCC_LSE_IsCSSDetected
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_IsCSSDetected(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_LSECSSD) == (RCC_BDCR_LSECSSD)) ? 1UL : 0UL);
}

/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EF_LSI LSI
  * @{
  */

/**
  * @brief  Enable LSI Oscillator
  * @rmtoll CSR          LSION         LL_RCC_LSI_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSI_Enable(void)
{
  SET_BIT(RCC->CSR, RCC_CSR_LSION);
}

/**
  * @brief  Disable LSI Oscillator
  * @rmtoll CSR          LSION         LL_RCC_LSI_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSI_Disable(void)
{
  CLEAR_BIT(RCC->CSR, RCC_CSR_LSION);
}

/**
  * @brief  Check if LSI is Ready
  * @rmtoll CSR          LSIRDY        LL_RCC_LSI_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSI_IsReady(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_LSIRDY) == (RCC_CSR_LSIRDY)) ? 1UL : 0UL);
}
/**
  * @}
  */
// todo
#if defined(RCC_BDCR_LSCSEL)
/** @defgroup RCC_LL_EF_LSC LSC
  * @{
  */
/**
  * @brief  Configure Low speed clock selection
  * @rmtoll BDCR         LSCSEL       LL_RCC_LSC_SetSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSE
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSC_SetSource(uint32_t Source)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSCSEL, Source);
}

/**
  * @brief  Get Low speed clock selection
  * @rmtoll BDCR         LSCSEL       LL_RCC_LSC_GetSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSE
  */
__STATIC_INLINE uint32_t LL_RCC_LSC_GetSource(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_LSCSEL));
}
/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EF_System System
  * @{
  */

/**
  * @brief  Configure the system clock source
  * @rmtoll CFGR         SW            LL_RCC_SetSysClkSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_HSISYS
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_HSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetSysClkSource(uint32_t Source)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_SW, Source);
}

/**
  * @brief  Get the system clock source
  * @rmtoll CFGR         SWS           LL_RCC_GetSysClkSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_HSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_LSI
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetSysClkSource(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_SWS));
}

/**
  * @brief  Set AHB prescaler
  * @rmtoll CFGR         HPRE          LL_RCC_SetAHBPrescaler
  * @param  Prescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetAHBPrescaler(uint32_t Prescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_HPRE, Prescaler);
}

/**
  * @brief  Set APB1 prescaler
  * @rmtoll CFGR         PPRE         LL_RCC_SetAPB1Prescaler
  * @param  Prescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetAPB1Prescaler(uint32_t Prescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_PPRE, Prescaler);
}

/**
  * @brief  Set HSI division factor
  * @rmtoll CR         HSIDIV          LL_RCC_SetHSIDiv
  * @note  HSIDIV parameter is only applied to SYSCLK_Frequency when HSI is used as
  * system clock source.
  * @param  HSIDiv  This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetHSIDiv(uint32_t HSIDiv)
{
  MODIFY_REG(RCC->CR, RCC_CR_HSIDIV, HSIDiv);
}
/**
  * @brief  Get AHB prescaler
  * @rmtoll CFGR         HPRE          LL_RCC_GetAHBPrescaler
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  */
__STATIC_INLINE uint32_t LL_RCC_GetAHBPrescaler(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_HPRE));
}

/**
  * @brief  Get APB1 prescaler
  * @rmtoll CFGR         PPRE         LL_RCC_GetAPB1Prescaler
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  */
__STATIC_INLINE uint32_t LL_RCC_GetAPB1Prescaler(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_PPRE));
}

/**
  * @brief  Get HSI division factor
  * @rmtoll CR         HSIDIV         LL_RCC_GetHSIDiv
  * @note  HSIDIV parameter is only applied to SYSCLK_Frequency when HSI is used as
  * system clock source.
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  */
__STATIC_INLINE uint32_t LL_RCC_GetHSIDiv(void)
{
  return (uint32_t)(READ_BIT(RCC->CR, RCC_CR_HSIDIV));
}
/**
  * @}
  */

/** @defgroup RCC_LL_EF_MCO MCO
  * @{
  */

/**
  * @brief  Configure MCOx
  * @rmtoll CFGR         MCOSEL        LL_RCC_ConfigMCO\n
  *         CFGR         MCOPRE        LL_RCC_ConfigMCO
  * @param  MCOxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCOSOURCE_NOCLOCK
  *         @arg @ref LL_RCC_MCOSOURCE_SYSCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HSI10M
  *         @arg @ref LL_RCC_MCOSOURCE_HSI
  *         @arg @ref LL_RCC_MCOSOURCE_HSE
  *         @arg @ref LL_RCC_MCOSOURCE_LSI
  *         @arg @ref LL_RCC_MCOSOURCE_LSE
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK_SRC
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK_UNGATE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @param  MCOxPrescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_DIV_1
  *         @arg @ref LL_RCC_MCO_DIV_2
  *         @arg @ref LL_RCC_MCO_DIV_4
  *         @arg @ref LL_RCC_MCO_DIV_8
  *         @arg @ref LL_RCC_MCO_DIV_16
  *         @arg @ref LL_RCC_MCO_DIV_32
  *         @arg @ref LL_RCC_MCO_DIV_64
  *         @arg @ref LL_RCC_MCO_DIV_128
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ConfigMCO(uint32_t MCOxSource, uint32_t MCOxPrescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_MCOSEL | RCC_CFGR_MCOPRE, MCOxSource | MCOxPrescaler);
}

/**
  * @brief  Get MCO clock source
  * @rmtoll CFGR         MCOSEL       LL_RCC_GetMCOClockSource
  * @param  MCOx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_MCOSOURCE_NOCLOCK
  *         @arg @ref LL_RCC_MCOSOURCE_SYSCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HSI10M
  *         @arg @ref LL_RCC_MCOSOURCE_HSI
  *         @arg @ref LL_RCC_MCOSOURCE_HSE
  *         @arg @ref LL_RCC_MCOSOURCE_LSI
  *         @arg @ref LL_RCC_MCOSOURCE_LSE
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK_SRC
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK_UNGATE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetMCOClockSource(uint32_t MCOx)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, MCOx));
}

/**
  * @brief  Get MCO division factor
  * @rmtoll CFGR         MCOPRE        LL_RCC_GetMCODiv
  * @param  MCOx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_MCO_DIV_1
  *         @arg @ref LL_RCC_MCO_DIV_2
  *         @arg @ref LL_RCC_MCO_DIV_4
  *         @arg @ref LL_RCC_MCO_DIV_8
  *         @arg @ref LL_RCC_MCO_DIV_16
  *         @arg @ref LL_RCC_MCO_DIV_32
  *         @arg @ref LL_RCC_MCO_DIV_64
  *         @arg @ref LL_RCC_MCO_DIV_128
  */
__STATIC_INLINE uint32_t LL_RCC_GetMCODiv(uint32_t MCOx)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_MCOPRE));
}

/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EF_Peripheral_Clock_Source Peripheral Clock Source
  * @{
  */
#if defined(RCC_CCIPR_TIMCLK_CTRL)
/**
  * @brief  Configure the TIMER PCLK frequency control.
  * @rmtoll CCIPR        TIMCLK_CTRL     LL_RCC_SetTIM_PCLK
  * @param  TIMPCLK_MULx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_TIMPCLK_MUL2
  *         @arg @ref LL_RCC_TIMPCLK_MUL1
  * @note   Depending on devices and packages,some TIM may not be available.
  *         Refer to device datasheet for TIM availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetTIM_PCLK(uint32_t TIMPCLK_MULx)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_TIMCLK_CTRL, TIMPCLK_MULx);
}
// todo
/**
  * @brief  Get the TIMPCLK multiple frequency factor.
  * @rmtoll CCIPR        TIMCLK_CTRL     LL_RCC_GetTIMPCLK_MUL
  * @param  TIMPCLK_CTRL This parameter can be one of the following values:
  *         @arg @ref LL_RCC_TIMPCLK_CTRL
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_TIMPCLK_MUL2
  *         @arg @ref LL_RCC_TIMPCLK_MUL1
  * @note   Depending on devices and packages,some TIM may not be available.
  *         Refer to device datasheet for TIM availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetTIMPCLK_MUL(uint32_t TIMPCLK_CTRL)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_TIMCLK_CTRL));
}
#endif /* RCC_CCIPR_TIMCLK_CTRL */

#if defined(COMP1)
/**
  * @brief  Configure COMPx clock source
  * @rmtoll CCIPR        COMPxSEL     LL_RCC_SetCOMPClockSource
  * @param  COMPxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_LSC
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_LSC
  * @note   Depending on devices and packages,some COMP may not be available.
  *         Refer to device datasheet for COMP availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetCOMPClockSource(uint32_t COMPxSource)
{
  MODIFY_REG(RCC->CCIPR, (COMPxSource & 0x0000FF00U), ((COMPxSource & 0xFFU) << 8U));
}

/**
  * @brief  Get COMPx clock source
  * @rmtoll CCIPR        COMPxSEL     LL_RCC_GetCOMPClockSource
  * @param  COMPx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_LSC
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_LSC
  * @note   Depending on devices and packages,some COMP may not be available.
  *         Refer to device datasheet for COMP availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetCOMPClockSource(uint32_t COMPx)
{
  return (uint32_t)((READ_BIT(RCC->CCIPR, COMPx) >> 8U) | COMPx);
}
#endif /* COMP1 */
// todo
#if defined(RCC_CCIPR_IWDGSEL)
/**
  * @brief  Configure IWDG clock source
  * @rmtoll CCIPR        IWDGSEL     LL_RCC_SetIWDGClockSource
  * @param  IWDGxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetIWDGClockSource(uint32_t IWDGxSource)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_IWDGSEL, IWDGxSource);
}
// todo
/**
  * @brief  Get IWDG clock source
  * @rmtoll CCIPR        IWDGSEL     LL_RCC_GetIWDGClockSource
  * @param  IWDGx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetIWDGClockSource(uint32_t IWDGx)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_IWDGSEL));
}
#endif /* RCC_CCIPR_IWDGSEL */

/**
  * @}
  */
#if defined(RTC)

/** @defgroup RCC_LL_EF_RTC RTC
  * @{
  */
#if defined(RCC_BDCR_RTCSEL)

/**
  * @brief  Set RTC Clock Source
  * @note Once the RTC clock source has been selected, it cannot be changed anymore unless
  *       the Backup domain is reset, or unless a failure is detected on LSE (LSECSSD is
  *       set). The BDRST bit can be used to reset them.
  * @rmtoll BDCR         RTCSEL        LL_RCC_SetRTCClockSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_NONE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV32
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV128
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV8
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetRTCClockSource(uint32_t Source)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_RTCSEL | RCC_BDCR_RTC_HSEDIV_SEL, Source);
}

/**
  * @brief  Get RTC Clock Source
  * @rmtoll BDCR         RTCSEL        LL_RCC_GetRTCClockSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_NONE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV32
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV128
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV8
  */
__STATIC_INLINE uint32_t LL_RCC_GetRTCClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_RTCSEL | RCC_BDCR_RTC_HSEDIV_SEL));
}
#endif
/**
  * @brief  Enable RTC
  * @rmtoll BDCR         RTCEN         LL_RCC_EnableRTC
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableRTC(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_RTCEN);
}

/**
  * @brief  Disable RTC
  * @rmtoll BDCR         RTCEN         LL_RCC_DisableRTC
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableRTC(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_RTCEN);
}

/**
  * @brief  Check if RTC has been enabled or not
  * @rmtoll BDCR         RTCEN         LL_RCC_IsEnabledRTC
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledRTC(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_RTCEN) == (RCC_BDCR_RTCEN)) ? 1UL : 0UL);
}

/**
  * @brief  Force the Backup domain reset
  * @rmtoll BDCR         BDRST         LL_RCC_ForceBackupDomainReset
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ForceBackupDomainReset(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_BDRST);
}

/**
  * @brief  Release the Backup domain reset
  * @rmtoll BDCR         BDRST         LL_RCC_ReleaseBackupDomainReset
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ReleaseBackupDomainReset(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_BDRST);
}

/**
  * @}
  */
#endif
// todo
/** @defgroup RCC_LL_EF_FLAG_Management FLAG Management
  * @{
  */

/**
  * @brief  Clear LSI ready interrupt flag
  * @rmtoll CICR         LSIRDYC       LL_RCC_ClearFlag_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSIRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSIRDYC);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Clear LSE ready interrupt flag
  * @rmtoll CICR         LSERDYC       LL_RCC_ClearFlag_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSERDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSERDYC);
}
#endif
/**
  * @brief  Clear HSI ready interrupt flag
  * @rmtoll CICR         HSIRDYC       LL_RCC_ClearFlag_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSIRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_HSIRDYC);
}

/**
  * @brief  Clear HSE ready interrupt flag
  * @rmtoll CICR         HSERDYC       LL_RCC_ClearFlag_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSERDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_HSERDYC);
}
// todo
/**
  * @brief  Clear Clock security system interrupt flag
  * @rmtoll CICR         CSSC          LL_RCC_ClearFlag_HSECSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSECSS(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_CSSC);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Clear LSE Clock security system interrupt flag
  * @rmtoll CICR         LSECSSC       LL_RCC_ClearFlag_LSECSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSECSS(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSECSSC);
}
#endif

/**
  * @brief  Check if LSI ready interrupt occurred or not
  * @rmtoll CIFR         LSIRDYF       LL_RCC_IsActiveFlag_LSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSIRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSIRDYF) == (RCC_CIFR_LSIRDYF)) ? 1UL : 0UL);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Check if LSE ready interrupt occurred or not
  * @rmtoll CIFR         LSERDYF       LL_RCC_IsActiveFlag_LSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSERDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSERDYF) == (RCC_CIFR_LSERDYF)) ? 1UL : 0UL);
}
#endif
/**
  * @brief  Check if HSI ready interrupt occurred or not
  * @rmtoll CIFR         HSIRDYF       LL_RCC_IsActiveFlag_HSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSIRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_HSIRDYF) == (RCC_CIFR_HSIRDYF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if HSE ready interrupt occurred or not
  * @rmtoll CIFR         HSERDYF       LL_RCC_IsActiveFlag_HSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSERDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_HSERDYF) == (RCC_CIFR_HSERDYF)) ? 1UL : 0UL);
}
// todo
/**
  * @brief  Check if Clock security system interrupt occurred or not
  * @rmtoll CIFR         CSSF          LL_RCC_IsActiveFlag_HSECSS
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSECSS(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_CSSF) == (RCC_CIFR_CSSF)) ? 1UL : 0UL);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Check if LSE Clock security system interrupt occurred or not
  * @rmtoll CIFR         LSECSSF       LL_RCC_IsActiveFlag_LSECSS
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSECSS(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSECSSF) == (RCC_CIFR_LSECSSF)) ? 1UL : 0UL);
}
#endif

/**
  * @brief  Check if RCC flag Independent Watchdog reset is set or not.
  * @rmtoll CSR          IWDGRSTF      LL_RCC_IsActiveFlag_IWDGRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_IWDGRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_IWDGRSTF) == (RCC_CSR_IWDGRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Option byte reset is set or not.
  * @rmtoll CSR          OBLRSTF       LL_RCC_IsActiveFlag_OBLRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_OBLRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_OBLRSTF) == (RCC_CSR_OBLRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Pin reset is set or not.
  * @rmtoll CSR          PINRSTF       LL_RCC_IsActiveFlag_PINRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_PINRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_PINRSTF) == (RCC_CSR_PINRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Software reset is set or not.
  * @rmtoll CSR          SFTRSTF       LL_RCC_IsActiveFlag_SFTRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_SFTRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_SFTRSTF) == (RCC_CSR_SFTRSTF)) ? 1UL : 0UL);
}
// todo
/**
  * @brief  Check if RCC flag BOR reset is set or not.
  * @rmtoll CSR          BORRSTF       LL_RCC_IsActiveFlag_BORRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_BORRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_BORRSTF) == (RCC_CSR_BORRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag POR reset is set or not.
  * @rmtoll CSR          PORRSTF       LL_RCC_IsActiveFlag_PORRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_PORRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_PORRSTF) == (RCC_CSR_PORRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Set RMVF bit to clear the reset flags.
  * @rmtoll CSR          RMVF          LL_RCC_ClearResetFlags
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearResetFlags(void)
{
  SET_BIT(RCC->CSR, RCC_CSR_RMVF);
}

/**
  * @brief  Enable NRST filter
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_EnableNRSTFilter
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableNRSTFilter(void)
{
  CLEAR_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS);
}

/**
  * @brief  Disable NRST filter
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_DisableNRSTFilter
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableNRSTFilter(void)
{
  SET_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS);
}

/**
  * @brief  Check if NRST filter is enable
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_IsEnableNRSTFilter
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnableNRSTFilter(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS) == (RCC_CSR_NRST_FLTDIS)) ? 0UL : 1UL);
}

/**
  * @}
  */

/** @defgroup RCC_LL_EF_IT_Management IT Management
  * @{
  */

/**
  * @brief  Enable LSI ready interrupt
  * @rmtoll CIER         LSIRDYIE      LL_RCC_EnableIT_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_LSIRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_LSIRDYIE);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Enable LSE ready interrupt
  * @rmtoll CIER         LSERDYIE      LL_RCC_EnableIT_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_LSERDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_LSERDYIE);
}
#endif
/**
  * @brief  Enable HSI ready interrupt
  * @rmtoll CIER         HSIRDYIE      LL_RCC_EnableIT_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_HSIRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_HSIRDYIE);
}

/**
  * @brief  Enable HSE ready interrupt
  * @rmtoll CIER         HSERDYIE      LL_RCC_EnableIT_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_HSERDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_HSERDYIE);
}
// todo
/**
  * @brief  Disable LSI ready interrupt
  * @rmtoll CIER         LSIRDYIE      LL_RCC_DisableIT_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_LSIRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_LSIRDYIE);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Disable LSE ready interrupt
  * @rmtoll CIER         LSERDYIE      LL_RCC_DisableIT_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_LSERDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_LSERDYIE);
}
#endif
/**
  * @brief  Disable HSI ready interrupt
  * @rmtoll CIER         HSIRDYIE      LL_RCC_DisableIT_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_HSIRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_HSIRDYIE);
}

/**
  * @brief  Disable HSE ready interrupt
  * @rmtoll CIER         HSERDYIE      LL_RCC_DisableIT_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_HSERDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_HSERDYIE);
}
// todo
/**
  * @brief  Checks if LSI ready interrupt source is enabled or disabled.
  * @rmtoll CIER         LSIRDYIE      LL_RCC_IsEnabledIT_LSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_LSIRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_LSIRDYIE) == (RCC_CIER_LSIRDYIE)) ? 1UL : 0UL);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Checks if LSE ready interrupt source is enabled or disabled.
  * @rmtoll CIER         LSERDYIE      LL_RCC_IsEnabledIT_LSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_LSERDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_LSERDYIE) == (RCC_CIER_LSERDYIE)) ? 1UL : 0UL);
}
#endif
/**
  * @brief  Checks if HSI ready interrupt source is enabled or disabled.
  * @rmtoll CIER         HSIRDYIE      LL_RCC_IsEnabledIT_HSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_HSIRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_HSIRDYIE) == (RCC_CIER_HSIRDYIE)) ? 1UL : 0UL);
}
// todo
/**
  * @brief  Checks if HSE ready interrupt source is enabled or disabled.
  * @rmtoll CIER         HSERDYIE      LL_RCC_IsEnabledIT_HSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_HSERDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_HSERDYIE) == (RCC_CIER_HSERDYIE)) ? 1UL : 0UL);
}
/**
  * @}
  */

#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_EF_Init De-initialization function
  * @{
  */
ErrorStatus LL_RCC_DeInit(void);
/**
  * @}
  */
// todo
/** @defgroup RCC_LL_EF_Get_Freq Get system and peripherals clocks frequency functions
  * @{
  */
void        LL_RCC_GetSystemClocksFreq(LL_RCC_ClocksTypeDef *RCC_Clocks);
uint32_t    LL_RCC_GetMCOClockFreq(uint32_t MCOx);
uint32_t    LL_RCC_GetLSCClockFreq(void);
#if defined(COMP1)
uint32_t    LL_RCC_GetCOMPClockFreq(uint32_t COMPx);
#endif
#if defined(RTC)
uint32_t    LL_RCC_GetRTCClockFreq(void);
#endif
/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/**
  * @}
  */

/**
  * @}
  */

#endif /* defined(RCC) */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __PY32T020_LL_RCC_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
